from astropy.io import fits
from astropy.wcs import WCS
from reproject import reproject_interp
import numpy as np
from glob import glob
from tqdm import tqdm


def main():
    drc_files = glob('*//*/HST/*/*drc.fits')
    ref = '/home/esheldon/data/RELHIC/MAST_2026-01-14T1403-F606W-1/MAST_2026-01-14T1403/HST/jfha01010/jfha01010_drc.fits'  # noqa

    # Define target WCS (use the first image as reference or define custom)
    # with fits.open(drc_files[0]) as hdul:
    with fits.open(ref) as hdul:
        ref_data = hdul['SCI'].data
        ref_wcs = WCS(hdul['SCI'].header)
        ref_shape = ref_data.shape

    # Initialize coadd arrays
    coadd = np.zeros(ref_shape, dtype=np.float32)
    weights = np.zeros(ref_shape, dtype=np.float32)

    for fname in tqdm(drc_files):
        with fits.open(fname) as hdul:
            sci_data = hdul['SCI'].data
            wht_data = (
                hdul['WHT'].data if 'WHT' in hdul else np.ones_like(sci_data)
            )  # Use weight if available
            hdr = hdul['SCI'].header
            # Reproject science and weight
            reproj_sci, footprint = reproject_interp(
                (sci_data, WCS(hdr)), ref_wcs, shape_out=ref_shape
            )
            reproj_wht, _ = reproject_interp(
                (wht_data, WCS(hdr)), ref_wcs, shape_out=ref_shape
            )
            # Accumulate
            coadd += reproj_sci * reproj_wht
            weights += reproj_wht

    # Final coadd: divide by weights (avoid division by zero)
    coadd_final = np.divide(coadd, weights, where=weights > 0)
    coadd_final[weights == 0] = np.nan

    # Save to FITS
    hdu = fits.PrimaryHDU(coadd_final, header=ref_wcs.to_header())
    hdu.writeto('coadd.fits', overwrite=True)


main()
